/*
 * ActionTask.h
 *
 * Created 8/23/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/23/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef ACTION_TASK_H
 #define ACTION_TASK_H
 
 template <typename T> class ActionTask;
 
 #include "global.h"
 
 #include "genericAsyncTask.h"
 #include "deletedChain.h"
 #include "asyncTaskManager.h"
 #include "ActionInfo.h"
 #include "Ralph.h"
 #include "StringConversion.h"
 
 /**
  * Class specification for ActionTask
  */
 template <typename T>
 class ActionTask : public GenericAsyncTask
 {
 // Data Members
 private:
    PT(Object<T>) _object_Ptr;
    PT(ActionInfo<T>) _action_info_Ptr;
    
 // Local Functions
 public:
    ActionTask( Object<T>* const object_Ptr, ActionInfo<T>* action_info_Ptr );
    //ActionTask( const string& name, ActionInfo<T>* action_info_Ptr );
    ActionTask( Object<T>* const object_Ptr, GenericAsyncTask::TaskFunc* function, ActionInfo<T>* action_info_Ptr, void* user_data = NULL );
    ActionTask( const ActionTask<T>& action_task );
    ALLOC_DELETED_CHAIN( ActionTask ); // Needed to accommodate additional data members (e.g. PT(ActionInfo<T>))
    virtual ~ActionTask();
    inline ActionTask<T>& operator=( const ActionTask<T>& action_task );
    virtual inline void add_to_task_manager();
    inline ActionInfo<T>* get_action_info();
    inline Object<T>* get_object();
    inline void set_object( Object<T>* const object_Ptr );
 
 // Protected Functions
 protected:
    inline void add_to_task_manager( AsyncTaskManager* task_mgr_Ptr ); // should probably move to protected
    
 // Static Functions
 public:
 
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  * It is assumed the name of the tasks associated with an Object is the Object's key.
  * See "Object.h" for clearing tasks and action tasks.
  */
 template <typename T>
 ActionTask<T>::ActionTask( Object<T>* const object_Ptr, ActionInfo<T>* action_info_Ptr )
               : GenericAsyncTask( StringConversion::to_str(object_Ptr->NodePath::get_key()) ),
                 _object_Ptr( object_Ptr ),
                 _action_info_Ptr( action_info_Ptr )
 {
    nassertv( _object_Ptr != NULL );
    nassertv( _action_info_Ptr != NULL );
 }
 
 /**
  * Alternative Constructor
  */
 //template <typename T>
 //ActionTask<T>::ActionTask( const string& name, ActionInfo<T>* action_info_Ptr )
 //              : GenericAsyncTask( name ),
 //                _action_info_Ptr( action_info_Ptr )
 //{
 //   nassertv( _action_info_Ptr != NULL );
 //   // GenericAsyncTask::set_name( StringConversion::to_str(_action_info_Ptr->get_object()->NodePath::get_key()) );
 //}
 
 /**
  * Alternative Constructor
  * It is assumed the name of the tasks associated with an Object is the Object's key.
  * See "Object.h" for clearing tasks and action tasks.
  */
 template <typename T>
 ActionTask<T>::ActionTask( Object<T>* const object_Ptr, GenericAsyncTask::TaskFunc* function, ActionInfo<T>* action_info_Ptr, void* user_data )
               : GenericAsyncTask( StringConversion::to_str(object_Ptr->NodePath::get_key()), function, user_data ),
                 _object_Ptr( object_Ptr ),
                 _action_info_Ptr( action_info_Ptr )
 {
    nassertv( _object_Ptr != NULL );
    nassertv( function != NULL );
    nassertv( _action_info_Ptr != NULL );
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 ActionTask<T>::ActionTask( const ActionTask<T>& action_task )
               : GenericAsyncTask( action_task ),
                 _object_Ptr( action_task._object_Ptr ),
                 _action_info_Ptr( action_task.action_info )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 ActionTask<T>::~ActionTask()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified ActionTask to this ActionTask.
  *
  * @param (const ActionTask<T>&) action_task
  * @return ActionTask<T>&
  */
 template <typename T>
 inline ActionTask<T>& ActionTask<T>::operator=( const ActionTask<T>& action_task )
 {
    GenericAsyncTask::operator=( action_task );
    _object_Ptr = action_task._object_Ptr;
    _action_info_Ptr = action_task.action_info;
    
    return *this;
 }
 
 /**
  * add_to_task_manager() adds this ActionTask to the AsyncTaskManager that handles actions.
  */
 template <typename T>
 inline void ActionTask<T>::add_to_task_manager()
 {
    nassertv( _action_info_Ptr != NULL );
    ActionTask<T>::add_to_task_manager( global::_action_task_mgr_Ptr );
 }
 
 /**
  * get_action_info() returns the a pointer to the ActionInfo
  * contained by this ActionTask.
  *
  * @return ActionInfo<T>*
  */
 template <typename T>
 inline ActionInfo<T>* ActionTask<T>::get_action_info()
 {
    return _action_info_Ptr;
 }
 
 /**
  * get_object() returns a pointer to the Object executing the
  * action described by this ActionTask.
  *
  * @return Object<T>*
  */
 template <typename T>
 inline Object<T>* ActionTask<T>::get_object()
 {
    return _object_Ptr;
 }
 
 /**
  * set_object() sets the Object pointer to the specified Object pointer.
  *
  * @param (Object<T>* const) object_Ptr
  */
 template <typename T>
 inline void ActionTask<T>::set_object( Object<T>* const object_Ptr )
 {
    nassertv( object_Ptr != NULL );
    
    _object_Ptr = object_Ptr;
 }
 
 /** PROTECTED FUNCTIONS **/
 
 /**
  * add_to_task_manager() adds this ActionTask to the specified AsyncTaskManager.
  */
 template <typename T>
 inline void ActionTask<T>::add_to_task_manager( AsyncTaskManager* task_mgr_Ptr )
 {
    nassertv( task_mgr_Ptr != NULL );
    nassertv( _action_info_Ptr != NULL );
    
    if ( GenericAsyncTask::get_function() != NULL )
    {
        task_mgr_Ptr->add( this );
    }
    else
    {
        _action_info_Ptr->split_tasks( this, task_mgr_Ptr );
    }
 }
 
 /** STATIC FUNCTIONS **/
 
 #endif // ACTION_TASK_H